<?php

/**
 * kiresu functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package kiresu
 */

if (! defined('_S_VERSION')) {
	// Replace the version number of the theme on each release.
	define('_S_VERSION', '1.0.0');
}

/**
 * Sets up theme defaults and registers support for various WordPress features.
 *
 * Note that this function is hooked into the after_setup_theme hook, which
 * runs before the init hook. The init hook is too late for some features, such
 * as indicating support for post thumbnails.
 */
function kiresu_setup()
{
	/*
		* Make theme available for translation.
		* Translations can be filed in the /languages/ directory.
		* If you're building a theme based on kiresu, use a find and replace
		* to change 'kiresu' to the name of your theme in all the template files.
		*/
	load_theme_textdomain('kiresu', get_template_directory() . '/languages');

	// Add default posts and comments RSS feed links to head.
	add_theme_support('automatic-feed-links');

	/*
		* Let WordPress manage the document title.
		* By adding theme support, we declare that this theme does not use a
		* hard-coded <title> tag in the document head, and expect WordPress to
		* provide it for us.
		*/
	add_theme_support('title-tag');

	/*
		* Enable support for Post Thumbnails on posts and pages.
		*
		* @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		*/
	add_theme_support('post-thumbnails');

	// This theme uses wp_nav_menu() in one location.
	// register_nav_menus(
	// 	array(
	// 		'menu-1' => esc_html__('Primary', 'kiresu'),
	// 	)
	// );

	/*
		* Switch default core markup for search form, comment form, and comments
		* to output valid HTML5.
		*/
	add_theme_support(
		'html5',
		array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
			'style',
			'script',
		)
	);

	// Set up the WordPress core custom background feature.
	add_theme_support(
		'custom-background',
		apply_filters(
			'kiresu_custom_background_args',
			array(
				'default-color' => 'ffffff',
				'default-image' => '',
			)
		)
	);

	// Add theme support for selective refresh for widgets.
	add_theme_support('customize-selective-refresh-widgets');

	/**
	 * Add support for core custom logo.
	 *
	 * @link https://codex.wordpress.org/Theme_Logo
	 */
	add_theme_support(
		'custom-logo',
		array(
			'height'      => 250,
			'width'       => 250,
			'flex-width'  => true,
			'flex-height' => true,
		)
	);
}
add_action('after_setup_theme', 'kiresu_setup');

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function kiresu_content_width()
{
	$GLOBALS['content_width'] = apply_filters('kiresu_content_width', 640);
}
add_action('after_setup_theme', 'kiresu_content_width', 0);

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function kiresu_widgets_init()
{
	register_sidebar(
		array(
			'name'          => esc_html__('Sidebar', 'kiresu'),
			'id'            => 'sidebar-1',
			'description'   => esc_html__('Add widgets here.', 'kiresu'),
			'before_widget' => '<section id="%1$s" class="widget %2$s">',
			'after_widget'  => '</section>',
			'before_title'  => '<h2 class="widget-title">',
			'after_title'   => '</h2>',
		)
	);
}
add_action('widgets_init', 'kiresu_widgets_init');

/**
 * Enqueue scripts and styles.
 */
function kiresu_scripts()
{
	// wp_enqueue_style('kiresu-style', get_stylesheet_uri(), array(), filemtime(get_template_directory() . '/style.css'));
	// wp_style_add_data('kiresu-style', 'rtl', 'replace');

	wp_enqueue_script('kiresu-navigation', get_template_directory_uri() . '/js/navigation.js', array(), _S_VERSION, true);

	if (is_singular() && comments_open() && get_option('thread_comments')) {
		wp_enqueue_script('comment-reply');
	}
}
add_action('wp_enqueue_scripts', 'kiresu_scripts');

/**
 * Implement the Custom Header feature.
 */
require get_template_directory() . '/inc/custom-header.php';

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/template-functions.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/customizer.php';

/**
 * Load Jetpack compatibility file.
 */
if (defined('JETPACK__VERSION')) {
	require get_template_directory() . '/inc/jetpack.php';
}

add_action('init', function () {
	add_post_type_support('page', 'page-attributes');
});

add_action('after_setup_theme', 'disable_admin_bar_for_non_admins');
function disable_admin_bar_for_non_admins()
{
	if (!current_user_can('administrator') && !is_admin()) {
		show_admin_bar(false);
	}
}

//Enable SVG upload
function smartwp_enable_svg_upload($mimes)
{
	//Only allow SVG upload by admins
	if (!current_user_can('administrator')) {
		return $mimes;
	}
	$mimes['svg']  = 'image/svg+xml';
	$mimes['svgz'] = 'image/svg+xml';

	return $mimes;
}
add_filter('upload_mimes', 'smartwp_enable_svg_upload');


//Disable XML-RPC
add_filter('xmlrpc_enabled', '__return_false');

function register_custom_menus()
{
	register_nav_menu('primary', __('Primary Menu'));
	register_nav_menu('primary-footer', __('Footer - Category'));
	register_nav_menu('footer-brands', __('Footer - Brands'));
	register_nav_menu('footer-about_us', __('Footer - About Us'));
	register_nav_menu('footer-privacy_policy', __('Footer - Privacy Policy'));
}
add_action('after_setup_theme', 'register_custom_menus');


function enqueue_custom_scripts()
{

	wp_enqueue_style(
		'owl-carousel',
		'https://cdnjs.cloudflare.com/ajax/libs/OwlCarousel2/2.3.4/assets/owl.carousel.min.css',
	);

	wp_enqueue_style(
		'owl-carousel-theme',
		'https://cdnjs.cloudflare.com/ajax/libs/OwlCarousel2/2.3.4/assets/owl.theme.default.min.css',
	);

	wp_enqueue_style(
		'tailwind',
		get_template_directory_uri() . '/css/tailwind.css',
		array(),
		filemtime(get_template_directory() . '/css/tailwind.css'),
	);

	wp_enqueue_style(
		'custom-fonts',
		get_template_directory_uri() . '/css/custom-fonts.css',
	);

	wp_enqueue_style(
		'custom-style',
		get_template_directory_uri() . '/css/custom-style.css',
		array(),
		filemtime(get_template_directory() . '/css/custom-style.css'),
	);

	// Custom Script
	wp_enqueue_script(
		'custom-script',
		get_template_directory_uri() . '/js/custom-script.js',
		array('jquery'),
		filemtime(get_template_directory() . '/js/custom-script.js'),
		true // Load in footer
	);

	// Pass JSON path to script
	wp_localize_script('custom-script', 'customSearchData', [
		'jsonUrl' => get_template_directory_uri() . '/assets/state-city.json'
	]);

	wp_localize_script('custom-script', 'ajax_object', [
		'ajax_url' => admin_url('admin-ajax.php')
	]);

	wp_enqueue_script(
		'owl-carousel-script',
		'https://cdnjs.cloudflare.com/ajax/libs/OwlCarousel2/2.3.4/owl.carousel.min.js',
		array('jquery'),
		null,
		true // Load in footer
	);

	wp_enqueue_style('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css');
	wp_enqueue_script('select2', 'https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js', ['jquery'], null, true);
}
add_action('wp_enqueue_scripts', 'enqueue_custom_scripts');


function allow_subscribers_to_upload_files()
{
	$role = get_role('subscriber');
	if ($role && !$role->has_cap('upload_files')) {
		$role->add_cap('upload_files');
	}
}
add_action('init', 'allow_subscribers_to_upload_files');

// Filter media library items for logged-in users (except admins)
add_filter('ajax_query_attachments_args', 'restrict_media_library_to_user', 10, 1);
function restrict_media_library_to_user($query)
{
	// Only apply to non-admin users
	if (!current_user_can('manage_options')) {
		$user_id = get_current_user_id();
		if ($user_id) {
			$query['author'] = $user_id;
		}
	}
	return $query;
}


function enqueue_state_city_script()
{
	wp_enqueue_script(
		'acf-state-city',
		get_template_directory_uri() . '/js/acf-state-city.js',
		['jquery'],
		filemtime(get_template_directory() . '/js/acf-state-city.js'),
		true
	);

	// Pass JSON path to script
	wp_localize_script('acf-state-city', 'acfStateCity', [
		'json_url' => get_template_directory_uri() . '/assets/state-city.json'
	]);
}
add_action('wp_enqueue_scripts', 'enqueue_state_city_script');


add_action('acf/save_post', 'set_company_post_title_and_content', 20);
function set_company_post_title_and_content($post_id)
{
	// Only target the 'company' custom post type
	if (get_post_type($post_id) !== 'company') {
		return;
	}

	// Get company name field
	$company_name = get_field('company_name', $post_id);
	$company_description = get_field('company_description', $post_id);

	if ($company_name) {

		$current_user_id = get_current_user_id();

		// Save the user ID to a post meta field like 'company_owner'
		update_post_meta($post_id, 'company_owner', $current_user_id);

		// Set post title and post name (slug)
		$post_data = array(
			'ID'         => $post_id,
			'post_title' => wp_strip_all_tags($company_name),
			'post_name'  => sanitize_title($company_name),
			'post_content' => wp_kses_post($company_description),
		);
		// Prevent infinite loop by temporarily unhooking this function
		remove_action('acf/save_post', 'set_company_post_title_and_content', 20);
		wp_update_post($post_data);
		add_action('acf/save_post', 'set_company_post_title_and_content', 20);
	}
}

add_filter('acf/load_field/name=state', 'load_indian_states');
function load_indian_states($field)
{
	$json_path = get_template_directory() . '/assets/state-city.json';
	if (file_exists($json_path)) {
		$data = json_decode(file_get_contents($json_path), true);
		if (is_array($data)) {
			$field['choices'] = array();
			foreach ($data as $state => $cities) {
				$field['choices'][$state] = $state;
			}
		}
	}
	return $field;
}

add_filter('acf/load_field/name=city', 'load_all_indian_cities');
function load_all_indian_cities($field)
{
	$json_path = get_template_directory() . '/assets/state-city.json';
	if (file_exists($json_path)) {
		$data = json_decode(file_get_contents($json_path), true);
		if (is_array($data)) {
			$field['choices'] = array();
			foreach ($data as $state => $cities) {
				foreach ($cities as $city) {
					$field['choices'][$city] = $city;
				}
			}
		}
	}
	return $field;
}


class Custom_Dropdown_Walker extends Walker_Nav_Menu
{
	// Start Level (submenu wrapper)
	function start_lvl(&$output, $depth = 0, $args = array())
	{
		$indent = str_repeat("\t", $depth);
		$output .= "\n$indent<div class=\"absolute left-0 mt-0 w-48 py-3 text-sm bg-white shadow-lg rounded hidden group-hover:block overflow-hidden\">\n<ul>\n";
	}

	// End Level
	function end_lvl(&$output, $depth = 0, $args = array())
	{
		$indent = str_repeat("\t", $depth);
		$output .= "$indent</ul></div>\n";
	}

	// Start Element
	function start_el(&$output, $item, $depth = 0, $args = array(), $id = 0)
	{
		$classes = empty($item->classes) ? array() : (array) $item->classes;
		$classes = array_filter($classes); // Remove empty class values

		// Add 'active' class if it's the current item or an ancestor
		if (
			in_array('current-menu-item', $classes) ||
			in_array('current_page_item', $classes) ||
			in_array('current-menu-ancestor', $classes) ||
			in_array('current-menu-parent', $classes)
		) {
			$classes[] = 'active';
		}
		if ($depth === 0 && in_array('menu-item-has-children', $classes)) {
			$classes[] = 'relative group';
		}
		$class_names = join(' ', apply_filters('nav_menu_css_class', $classes, $item, $args, $depth));
		$li_class = esc_attr($class_names);
		$output .= '<li class="' . $li_class . '">';

		$atts = array();
		$atts['href'] = !empty($item->url) ? $item->url : '#';
		$attributes = '';
		foreach ($atts as $attr => $value) {
			$attributes .= ' ' . $attr . '="' . esc_attr($value) . '"';
		}

		if ($depth === 0 && in_array('menu-item-has-children', $classes)) {
			$output .= '<button class="flex items-center border-0 text-black">' . $item->title;
			$output .= '<svg class="ml-1 w-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 9l-7 7-7-7" />
                        </svg></button>';
		} else {
			$output .= '<a class="block py-2 px-3 text-black hover:bg-primary" ' . $attributes . '>' . $item->title . '</a>';
		}
	}

	// End Element
	function end_el(&$output, $item, $depth = 0, $args = array())
	{
		$output .= "</li>\n";
	}
}

class Mobile_Dropdown_Walker extends Walker_Nav_Menu
{
	function start_lvl(&$output, $depth = 0, $args = array())
	{
		$output .= '<ul class="hidden list-none text-sm">' . "\n";
	}

	function end_lvl(&$output, $depth = 0, $args = array())
	{
		$output .= "</ul>\n";
	}

	function start_el(&$output, $item, $depth = 0, $args = array(), $id = 0)
	{
		$has_children = in_array('menu-item-has-children', (array) $item->classes);
		$classes = empty($item->classes) ? array() : (array) $item->classes;
		$classes = array_filter($classes); // Remove empty class values

		// Add 'active' class if it's the current item or an ancestor
		if (
			in_array('current-menu-item', $classes) ||
			in_array('current_page_item', $classes) ||
			in_array('current-menu-ancestor', $classes) ||
			in_array('current-menu-parent', $classes)
		) {
			$classes[] = 'active';
		}
		if ($depth === 0) {
			$classes[] = 'relative group';
		}
		$class_names = join(' ', apply_filters('nav_menu_css_class', $classes, $item, $args, $depth));
		$li_class = esc_attr($class_names);

		$output .= '<li class="' . $li_class . '">' . "\n";

		$title = $item->title;
		$url = esc_url($item->url);

		if ($depth === 0 && $has_children) {
			$output .= '<button class="flex justify-between items-center w-full text-left text-black submenu-toggle">';
			$output .= $title;
			$output .= '<svg class="w-4 h-4 transform transition-transform" viewBox="0 0 24 24">
                            <path d="M6 9l6 6 6-6" stroke="currentColor" stroke-width="2" fill="none" />
                        </svg>';
			$output .= '</button>';
		} elseif ($depth === 0) {
			$output .= '<a href="' . $url . '" class="block py-1 text-black hover:bg-primary">' . $title . '</a>';
		} else {
			$output .= '<a href="' . $url . '" class="block px-3 py-1 text-black hover:bg-primary">' . $title . '</a>';
		}
	}

	function end_el(&$output, $item, $depth = 0, $args = array())
	{
		$output .= "</li>\n";
	}
}


function custom_search_form_shortcode()
{
	ob_start();
?>
	<form role="search" method="get" action="<?php echo esc_url(home_url('/')); ?>">
		<?php
		$search_val = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
		?>
		<div class="mx-auto py-6 xl:py-8 px-6 xl:px-10 rounded-xl shadow-lg bg-primary">
			<h2 class="text-2xl font-bold mb-3 text-center md:text-start">Discover here</h2>
			<div class="flex flex-wrap items-center justify-center font-medium space-y-3 md:space-y-0 md:space-x-3">
				<div class="md:flex-1 w-full md:w-auto">
					<input type="text" name="s" placeholder="What are you looking for?" class="w-full h-11 text-black px-4 py-2 border border-white rounded-md focus:outline-none placeholder:text-black" value="<?php echo esc_attr($search_val); ?>">
				</div>
				<div class="w-full md:w-[10em] lg:w-[14em]">
					<select name="city" id="city-select" class="w-full h-11 px-4 border border-white rounded-md focus:outline-none text-black bg-white">
						<option value="">Select City</option>
					</select>
				</div>
				<button type="submit" class="h-11 rounded-md btn btn-transparent hover:border-black focus:outline-none focus:shadow-outline">Find</button>
			</div>
		</div>

	</form>
<?php
	return ob_get_clean();
}
add_shortcode('custom_search_form', 'custom_search_form_shortcode');

function icd_override_ppp_for_category_archieve($query)
{
	// Only target the main query on the frontend and product category pages
	if (!is_admin() && $query->is_main_query() && (is_tax('item_category') || is_post_type_archive('company'))) {
		// Change number of products per page
		$query->set('posts_per_page', 6);
	}
}
add_action('pre_get_posts', 'icd_override_ppp_for_category_archieve');



// 1. Register endpoint
function ur_add_custom_account_endpoint()
{
	add_rewrite_endpoint('my-company', EP_ROOT | EP_PAGES);
}
add_action('init', 'ur_add_custom_account_endpoint');

// 2. Flag custom tab as active
function ur_mark_custom_tab_active()
{
	global $wp_query;

	if (isset($wp_query->query_vars['my-company'])) {
		$GLOBALS['ur_custom_tab_active'] = true;
	}
}
add_action('template_redirect', 'ur_mark_custom_tab_active');

// 3. Remove dashboard content if on custom tab
function ur_maybe_remove_dashboard_content()
{
	if (! empty($GLOBALS['ur_custom_tab_active'])) {
		remove_all_actions('user_registration_account_content', 10);
	}
}
add_action('user_registration_account_content', 'ur_maybe_remove_dashboard_content', 5);

// 4. Display custom tab content
function ur_display_custom_tab_content()
{
	if (!empty($GLOBALS['ur_custom_tab_active']) && is_user_logged_in()) {
		$user_id = get_current_user_id();

		// Fetch user's company
		$args = array(
			'post_type'    => 'company',
			'post_status'  => array('pending', 'publish'),
			'meta_key'     => 'company_owner',
			'meta_value'   => $user_id,
			'numberposts'  => 1,
		);

		$company_posts = get_posts($args);

		if (empty($company_posts)) {
			// No company registered
			echo '<h2>No Company Registered</h2>';
			echo "<p>You haven't registered your company yet.</p>";
			echo '<a href="' . esc_url(site_url('/company-registration')) . '" class="btn bg-primary">Register Now</a>';
		} else {
			$company = $company_posts[0];
			$status = get_post_status($company);
			$date_registered = get_the_date('d/m/Y', $company);
			$company_title = get_the_title($company);
			$contact_url = site_url('/contact');
			$delete_url = wp_nonce_url(
				add_query_arg([
					'action'    => 'delete_company',
					'post_id'   => $company->ID,
				], site_url($_SERVER['REQUEST_URI'])),
				'delete_company_' . $company->ID
			);

			// Display in a table
			echo '<style>.ur-company-table {
				width: 100%;
				border-collapse: collapse;
				margin-top: 20px;
			}

			.ur-company-table th,
			.ur-company-table td {
				padding: 12px 16px;
				border: 1px solid #ddd;
				text-align: left;
			}

			.ur-company-table th {
				background-color: #f8f8f8;
				font-weight: 600;
			}

			.ur-company-table tr:nth-child(even) {
				background-color: #f9f9f9;
			}

			.ur-company-table a.button {
				padding: 6px 12px;
				background-color: #0073aa;
				color: #fff;
				text-decoration: none;
				border-radius: 3px;
				display: inline-block;
			}

			.ur-company-table a.button:hover {
				background-color: #005177;
			}
			.ur-company-table .company__link{
				font-weight: 500;
				text-decoration: underline;	
			}
			</style>';
			echo '<h2>Your Company</h2>';
			echo '<div class="my-company-table__main">';
			echo '<table class="wp-list-table widefat fixed striped ur-company-table">';
			echo '<thead><tr>
					<th>Name</th>
					<th>Date of Registration</th>
					<th>Status</th>
					<th>Action</th>
				  </tr></thead>';
			echo '<tbody>';
			echo '<tr>';
			// Name (linked if published)
			if ($status === 'publish') {
				echo '<td><a class="company__link" href="' . esc_url(get_permalink($company)) . '" target="_blank">' . esc_html($company_title) . '</a></td>';
			} else {
				echo '<td>' . esc_html($company_title) . '</td>';
			}

			echo '<td>' . esc_html($date_registered) . '</td>';
			echo '<td>' . ucfirst($status) . '</td>';
			echo '<td>
					<a href="' . esc_url($contact_url) . '" class="btn bg-primary">Delete</a>
				</td>';
			echo '</tr>';
			echo '</tbody></table>';
			echo '</div>';
		}
	}
}
add_action('user_registration_account_content', 'ur_display_custom_tab_content', 20);



add_filter('user_registration_account_menu_items', 'ur_add_custom_tab_to_account_menu_positioned');
function ur_add_custom_tab_to_account_menu_positioned($items)
{
	$new_items = [];

	// Convert to indexed array to maintain order
	foreach ($items as $endpoint => $label) {
		$new_items[$endpoint] = $label;

		// Insert after the first item
		if (count($new_items) === 1) {
			$new_items['my-company'] = __('My Company', 'kiresu');
		}
	}

	return $new_items;
}


function custom_breadcrumbs_shortcode()
{
	global $post;

	// Get the current object (post, term, etc.)
	$title = '';
	$breadcrumbs = array();

	// Always start with Home
	$breadcrumbs[] = '<a href="' . home_url() . '" class="hover:underline">Home</a>';

	if (is_singular()) {
		$post_type = get_post_type();

		// Add parent archive if post type is not 'post' or 'page'
		if ($post_type && $post_type !== 'post' && $post_type !== 'page') {
			$post_type_object = get_post_type_object($post_type);
			if ($post_type_object && $post_type_object->has_archive) {
				if (strtolower($post_type_object->labels->name) == 'companies') {
					$post_type_obj_name = 'Company';
				} else {
					$post_type_obj_name = $post_type_object->labels->name;
				}
				$breadcrumbs[] = '<a href="' . get_post_type_archive_link($post_type) . '" class="hover:underline">' . $post_type_obj_name . '</a>';
			}
		}

		// If page has parent, include its link
		if (is_page() && $post->post_parent) {
			$parent_id = $post->post_parent;
			$breadcrumbs[] = '<a href="' . get_permalink($parent_id) . '" class="hover:underline">' . get_the_title($parent_id) . '</a>';
		}

		// Current post/page title
		$title = get_the_title();
	} elseif (is_category() || is_tag() || is_tax()) {
		$term = get_queried_object();
		if ($term && isset($term->name)) {
			$taxonomy = get_taxonomy($term->taxonomy);
			if ($taxonomy && $taxonomy->labels->name) {
				$breadcrumbs[] = '<a href="' . get_term_link($term) . '" class="hover:underline">' . $taxonomy->labels->name . '</a>';
			}
			$title = $term->name;
		}
	} elseif (is_post_type_archive()) {
		$post_type = get_post_type_object(get_post_type());
		$title = $post_type->labels->name;
	} elseif (is_home()) {
		$title = 'Blog';
	} elseif (is_search()) {
		$title = 'Search results for "' . get_search_query() . '"';
	} elseif (is_404()) {
		$title = '404';
	}

	// Combine breadcrumb trail
	$output  = '<div class="breadcrumb relative z-10 -mt-[1.5rem] mb-8 xl:mb-14">';
	$output .= '<div class="container">';
	$output .= '<nav class="bg-primary rounded-xl px-6 py-3 text-base font-medium text-black flex-wrap flex gap-x-3 gap-y-2">';
	$output .= implode('<span>»</span>', $breadcrumbs);
	if (!empty($title)) {
		$output .= '<span>»</span><span class="text-gray">' . esc_html($title) . '</span>';
	}
	$output .= '</nav></div></div>';

	return $output;
}
add_shortcode('custom_breadcrumbs', 'custom_breadcrumbs_shortcode');


function my_custom_admin_inline_css()
{
	echo '<style>
			#home__customer_reviews .acf-table .acf-row td:not(.remove) {
				border-bottom: 3px solid black;
			}
		</style>';
}
add_action('admin_head', 'my_custom_admin_inline_css');


add_action('wp_ajax_submit_review', 'handle_review_submission');

function handle_review_submission()
{
	if (! is_user_logged_in()) {
		wp_send_json_error('You must be logged in.');
	}

	$user_id = get_current_user_id();
	$full_name = sanitize_text_field($_POST['full_name'] ?? '');
	$rating = intval($_POST['rating'] ?? 0);
	$title = sanitize_text_field($_POST['your_title'] ?? '');
	$review_text = sanitize_textarea_field($_POST['reviewText'] ?? '');
	$company_id = intval($_POST['company_id'] ?? 0);
	$current_user_id = intval($_POST['current_user_id'] ?? 0);
	$company_post = get_post($company_id);
	$company_name = ($company_post && $company_post->post_status === 'publish') ? $company_post->post_title : '';

	if (empty($full_name) || empty($rating) || empty($title) || empty($review_text)) {
		wp_send_json_error('All fields are required.');
	}

	$review_id = wp_insert_post([
		'post_type'   => 'review',
		'post_title'  => $title,
		'post_content' => $review_text,
		'post_status' => 'publish',
		'post_author' => $user_id,
		'meta_input'  => [
			'reviewer_name' => $full_name,
			'rating' => $rating,
			'company_id' => $company_id,
			'company_name'  => $company_name,
			'current_user_id'  => $current_user_id,
		]
	]);

	if (is_wp_error($review_id)) {
		wp_send_json_error('Could not save review.');
	}

	wp_send_json_success('Review saved successfully!');
}


function filter_search_query($query)
{
	if (is_admin() || !$query->is_main_query() || !$query->is_search()) {
		return;
	}

	// Get search parameters
	$search_term = isset($_GET['s']) ? sanitize_text_field($_GET['s']) : '';
	$search_city = isset($_GET['city']) ? sanitize_text_field($_GET['city']) : '';

	$search_post_ids = [];
	$taxonomy_post_ids = [];

	// 1. Get post IDs by search term (title/content)
	if (!empty($search_term)) {
		$search_query = new WP_Query([
			'post_type'      => 'company',
			'post_status'    => 'publish',
			's'              => $search_term,
			'fields'         => 'ids',
			'posts_per_page' => -1,
		]);
		$search_post_ids = $search_query->posts;
	}

	// 2. Get post IDs from matching taxonomy terms
	if (!empty($search_term)) {
		$matching_terms = get_terms([
			'taxonomy'   => 'item_category',
			'hide_empty' => false,
			'name__like' => $search_term,
		]);

		if (!is_wp_error($matching_terms) && !empty($matching_terms)) {
			$term_ids = wp_list_pluck($matching_terms, 'term_id');

			$taxonomy_query = new WP_Query([
				'post_type'      => 'company',
				'post_status'    => 'publish',
				'tax_query'      => [
					[
						'taxonomy' => 'item_category',
						'field'    => 'term_id',
						'terms'    => $term_ids,
					]
				],
				'fields'         => 'ids',
				'posts_per_page' => -1,
			]);

			$taxonomy_post_ids = $taxonomy_query->posts;
		}
	}

	// 3. Merge post IDs
	$combined_ids = array_unique(array_merge($search_post_ids, $taxonomy_post_ids));

	// Apply to main query
	$query->set('post_type', 'company');
	$query->set('post_status', 'publish');
	$query->set('posts_per_page', 8);
	$query->set('post__in', !empty($combined_ids) ? $combined_ids : [0]);

	// City meta filter
	if (!empty($search_city)) {
		$query->set('meta_query', [[
			'key'     => 'city',
			'value'   => $search_city,
			'compare' => 'LIKE',
		]]);
	}
}
add_action('pre_get_posts', 'filter_search_query');


add_action('init', 'override_ur_profile_pic_upload');

function override_ur_profile_pic_upload()
{
	// Remove original AJAX callback
	remove_action('wp_ajax_user_registration_profile_pic_upload', array('UR_AJAX', 'profile_pic_upload'));
	remove_action('wp_ajax_nopriv_user_registration_profile_pic_upload', array('UR_AJAX', 'profile_pic_upload'));

	// Add your custom version
	add_action('wp_ajax_user_registration_profile_pic_upload', 'my_custom_profile_pic_upload');
	add_action('wp_ajax_nopriv_user_registration_profile_pic_upload', 'my_custom_profile_pic_upload');
}

function my_custom_profile_pic_upload()
{
	check_ajax_referer('user_registration_profile_picture_upload_nonce', 'security');

	$nonce = isset($_REQUEST['security']) ? sanitize_text_field(wp_unslash($_REQUEST['security'])) : false;

	$flag = wp_verify_nonce($nonce, 'user_registration_profile_picture_upload_nonce');

	if (true != $flag || is_wp_error($flag)) {
		wp_send_json_error(
			array(
				'message' => __('Nonce error, please reload.', 'user-registration'),
			)
		);
	}
	$user_id = get_current_user_id();

	if ($user_id <= 0) {
		return;
	}

	if (isset($_FILES['file']['size'])) {
		$file_size = intval($_FILES['file']['size']);
		$max_size = 2 * 1024 * 1024; // 2MB in bytes

		if ($file_size > $max_size) {
			wp_send_json_error(['message' => __('File exceeds 2MB size limit.', 'user-registration')]);
		}
	}

	if (isset($_FILES['file']['size']) && wp_unslash(sanitize_key($_FILES['file']['size']))) {
		if (! function_exists('wp_handle_upload')) {
			include_once ABSPATH . 'wp-admin/includes/file.php';
		}

		$upload = isset($_FILES['file']) ? $_FILES['file'] : array(); // phpcs:ignore

		// valid extension for image.
		$valid_extensions = 'image/jpeg,image/gif,image/png';
		$form_id          = ur_get_form_id_by_userid($user_id);

		if (class_exists('UserRegistrationAdvancedFields')) {
			$field_data       = ur_get_field_data_by_field_name($form_id, 'profile_pic_url');
			$valid_extensions = isset($field_data['advance_setting']->valid_file_type) ? implode(', ', $field_data['advance_setting']->valid_file_type) : $valid_extensions;
		}

		$valid_extension_type = explode(',', $valid_extensions);
		$valid_ext            = array();

		foreach ($valid_extension_type as $key => $value) {
			$image_extension   = explode('/', $value);
			$valid_ext[$key] = isset($image_extension[1]) ? $image_extension[1] : '';

			if ('jpeg' === $valid_ext[$key]) {
				$index               = count($valid_extension_type);
				$valid_ext[$index] = 'jpg';
			}
		}

		$src_file_name  = isset($upload['name']) ? $upload['name'] : '';
		$file_extension = strtolower(pathinfo($src_file_name, PATHINFO_EXTENSION));
		$file_mime_type = isset($upload['tmp_name']) ? mime_content_type($upload['tmp_name']) : '';

		if (! in_array($file_mime_type, $valid_extension_type)) {
			wp_send_json_error(
				array(
					'message' => __('Invalid file type, please contact with site administrator.', 'user-registration'),
				)
			);
		}
		// Validates if the uploaded file has the acceptable extension.
		if (! in_array($file_extension, $valid_ext)) {
			wp_send_json_error(
				array(
					'message' => __('Invalid file type, please contact with site administrator.', 'user-registration'),
				)
			);
		}

		$upload_path = ur_get_tmp_dir();

		// Checks if the upload directory has the write premission.
		if (! wp_is_writable($upload_path)) {
			wp_send_json_error(
				array(
					'message' => __('Upload path permission deny.', 'user-registration'),
				)
			);
		}
		$upload_path = $upload_path . '/';
		$file_name   = wp_unique_filename($upload_path, $upload['name']);
		$file_path   = $upload_path . sanitize_file_name($file_name);
		if (move_uploaded_file($upload['tmp_name'], $file_path)) {
			$files = array(
				'file_name'      => $file_name,
				'file_path'      => $file_path,
				'file_extension' => $file_extension,
			);

			$attachment_id = wp_rand();

			ur_clean_tmp_files();
			$url = UR_UPLOAD_URL . 'temp-uploads/' . sanitize_file_name($file_name);
			wp_send_json_success(
				array(
					'attachment_id' => $attachment_id,
					'upload_files'  => crypt_the_string(maybe_serialize($files), 'e'),
					'url'           => $url,
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => __('File cannot be uploaded.', 'user-registration'),
				)
			);
		}
	} elseif (isset($_FILES['file']['error']) && UPLOAD_ERR_NO_FILE !== $_FILES['file']['error']) {
		switch ($_FILES['file']['error']) {
			case UPLOAD_ERR_INI_SIZE:
				wp_send_json_error(
					array(
						'message' => __('File size exceed, please check your file size.', 'user-registration'),
					)
				);
				break;
			default:
				wp_send_json_error(
					array(
						'message' => __('Something went wrong while uploading, please contact your site administrator.', 'user-registration'),
					)
				);
				break;
		}
	}
}


function detect_user_registration_shortcode($posts) {
	if (empty($posts)) return $posts;

	foreach ($posts as $post) {
		if (has_shortcode($post->post_content, 'user_registration_my_account')) {
			add_filter('should_dequeue_ur_my_account_script', '__return_true');
			break;
		}
	}

	return $posts;
}
add_filter('the_posts', 'detect_user_registration_shortcode', 10);


function late_dequeue_and_replace_ur_my_account_script() {
	if (apply_filters('should_dequeue_ur_my_account_script', false)) {
		// Dequeue the original script
		wp_dequeue_script('ur-my-account');
		wp_deregister_script('ur-my-account');
		wp_enqueue_script('sweetalert2');

		// Enqueue your custom script
		wp_enqueue_script(
			'custom-ur-my-account',
			get_template_directory_uri() . '/js/my-account.js',
			array('jquery'),
			'1.0.0',
			true
		);
	}
}
add_action('wp_footer', 'late_dequeue_and_replace_ur_my_account_script', 1);

